import { users, quotationRequests, contactMessages, type User, type InsertUser, type QuotationRequest, type InsertQuotationRequest, type ContactMessage, type InsertContactMessage } from "@shared/schema";
import { db } from "./db";
import { eq } from "drizzle-orm";

export interface IStorage {
  getUser(id: string): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  createUser(insertUser: InsertUser): Promise<User>;
  createQuotationRequest(insertQuotationRequest: InsertQuotationRequest): Promise<QuotationRequest>;
  getQuotationRequests(): Promise<QuotationRequest[]>;
  createContactMessage(insertContactMessage: InsertContactMessage): Promise<ContactMessage>;
  getContactMessages(): Promise<ContactMessage[]>;
}

export class DatabaseStorage implements IStorage {
  async getUser(id: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || undefined;
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || undefined;
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    const [user] = await db
      .insert(users)
      .values(insertUser)
      .returning();
    return user;
  }

  async createQuotationRequest(insertQuotationRequest: InsertQuotationRequest): Promise<QuotationRequest> {
    const [quotation] = await db
      .insert(quotationRequests)
      .values(insertQuotationRequest)
      .returning();
    return quotation;
  }

  async getQuotationRequests(): Promise<QuotationRequest[]> {
    return await db.select().from(quotationRequests);
  }

  async createContactMessage(insertContactMessage: InsertContactMessage): Promise<ContactMessage> {
    const [message] = await db
      .insert(contactMessages)
      .values(insertContactMessage)
      .returning();
    return message;
  }

  async getContactMessages(): Promise<ContactMessage[]> {
    return await db.select().from(contactMessages);
  }
}

export const storage = new DatabaseStorage();