import type { Express } from "express";
import { createServer, type Server } from "http";
import { storage } from "./storage";
import { insertQuotationRequestSchema, insertContactMessageSchema } from "@shared/schema";
import { z } from "zod";

export async function registerRoutes(app: Express): Promise<Server> {
  // Quotation request endpoint
  app.post("/api/quotations", async (req, res) => {
    try {
      const validatedData = insertQuotationRequestSchema.parse(req.body);
      const quotation = await storage.createQuotationRequest(validatedData);
      res.status(201).json({ success: true, id: quotation.id });
    } catch (error) {
      if (error instanceof z.ZodError) {
        res.status(400).json({ success: false, error: "Validation failed", details: error.errors });
      } else {
        console.error("Error creating quotation request:", error);
        res.status(500).json({ success: false, error: "Failed to submit quotation request" });
      }
    }
  });

  // Contact message endpoint
  app.post("/api/contact", async (req, res) => {
    try {
      const validatedData = insertContactMessageSchema.parse(req.body);
      const message = await storage.createContactMessage(validatedData);
      res.status(201).json({ success: true, id: message.id });
    } catch (error) {
      if (error instanceof z.ZodError) {
        res.status(400).json({ success: false, error: "Validation failed", details: error.errors });
      } else {
        console.error("Error creating contact message:", error);
        res.status(500).json({ success: false, error: "Failed to send message" });
      }
    }
  });

  // Get quotations (for admin purposes)
  app.get("/api/quotations", async (req, res) => {
    try {
      const quotations = await storage.getQuotationRequests();
      res.json({ success: true, data: quotations });
    } catch (error) {
      console.error("Error fetching quotations:", error);
      res.status(500).json({ success: false, error: "Failed to fetch quotations" });
    }
  });

  // Get contact messages (for admin purposes)
  app.get("/api/contact", async (req, res) => {
    try {
      const messages = await storage.getContactMessages();
      res.json({ success: true, data: messages });
    } catch (error) {
      console.error("Error fetching contact messages:", error);
      res.status(500).json({ success: false, error: "Failed to fetch messages" });
    }
  });

  const httpServer = createServer(app);
  return httpServer;
}
