import { type User, type InsertUser, type QuotationRequest, type InsertQuotationRequest, type ContactMessage, type InsertContactMessage } from "@shared/schema";

export interface IStorage {
  getUser(id: string): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  createUser(insertUser: InsertUser): Promise<User>;
  createQuotationRequest(insertQuotationRequest: InsertQuotationRequest): Promise<QuotationRequest>;
  getQuotationRequests(): Promise<QuotationRequest[]>;
  createContactMessage(insertContactMessage: InsertContactMessage): Promise<ContactMessage>;
  getContactMessages(): Promise<ContactMessage[]>;
}

// Simple in-memory storage for testing without database setup
export class MemStorage implements IStorage {
  private users: User[] = [];
  private quotationRequests: QuotationRequest[] = [];
  private contactMessages: ContactMessage[] = [];

  async getUser(id: string): Promise<User | undefined> {
    return this.users.find(u => u.id === id);
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    return this.users.find(u => u.username === username);
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    const user: User = {
      id: Math.random().toString(36).substr(2, 9),
      ...insertUser,
    };
    this.users.push(user);
    return user;
  }

  async createQuotationRequest(insertQuotationRequest: InsertQuotationRequest): Promise<QuotationRequest> {
    const quotation: QuotationRequest = {
      id: Math.random().toString(36).substr(2, 9),
      createdAt: new Date(),
      companyName: insertQuotationRequest.companyName || null,
      cargoType: insertQuotationRequest.cargoType || null,
      weight: insertQuotationRequest.weight || null,
      details: insertQuotationRequest.details || null,
      newsletter: insertQuotationRequest.newsletter || false,
      fullName: insertQuotationRequest.fullName,
      email: insertQuotationRequest.email,
      phone: insertQuotationRequest.phone,
      serviceType: insertQuotationRequest.serviceType,
      origin: insertQuotationRequest.origin,
      destination: insertQuotationRequest.destination,
    };
    this.quotationRequests.push(quotation);
    return quotation;
  }

  async getQuotationRequests(): Promise<QuotationRequest[]> {
    return this.quotationRequests;
  }

  async createContactMessage(insertContactMessage: InsertContactMessage): Promise<ContactMessage> {
    const message: ContactMessage = {
      id: Math.random().toString(36).substr(2, 9),
      createdAt: new Date(),
      ...insertContactMessage,
    };
    this.contactMessages.push(message);
    return message;
  }

  async getContactMessages(): Promise<ContactMessage[]> {
    return this.contactMessages;
  }
}

export const storage = new MemStorage();