#!/bin/bash

# SiyanTransit Express Project Export Script
echo "Creating project export..."

# Create a timestamp for the export
TIMESTAMP=$(date +%Y%m%d_%H%M%S)
EXPORT_DIR="siyantransit_export_$TIMESTAMP"

# Create export directory
mkdir -p "$EXPORT_DIR"

# Copy main project files
echo "Copying project files..."

# Copy package.json and configuration files
cp package.json "$EXPORT_DIR/"
cp package-lock.json "$EXPORT_DIR/"
cp tsconfig.json "$EXPORT_DIR/"
cp vite.config.ts "$EXPORT_DIR/"
cp tailwind.config.ts "$EXPORT_DIR/"
cp postcss.config.js "$EXPORT_DIR/"
cp components.json "$EXPORT_DIR/"
cp drizzle.config.ts "$EXPORT_DIR/"

# Copy client folder
cp -r client "$EXPORT_DIR/"

# Copy server folder
cp -r server "$EXPORT_DIR/"

# Copy shared folder
cp -r shared "$EXPORT_DIR/"

# Copy important documentation
cp replit.md "$EXPORT_DIR/" 2>/dev/null || echo "replit.md not found, skipping..."

# Create a README for the export
cat > "$EXPORT_DIR/README.md" << 'EOF'
# SiyanTransit Express - Logistics Website

This is a professional logistics company website built with Node.js, Express, React, and PostgreSQL.

## Features
- Complete logistics company website
- Quotation request system
- Contact form management
- PostgreSQL database integration
- Responsive design with modern UI
- Professional blue and orange color scheme

## Project Structure
- `client/` - React frontend application
- `server/` - Express.js backend with API routes
- `shared/` - Common schemas and types
- Database: PostgreSQL with Drizzle ORM

## Setup Instructions
1. Install Node.js (v20 or higher)
2. Run `npm install` to install dependencies
3. Set up PostgreSQL database and configure DATABASE_URL
4. Run `npm run db:push` to create database tables
5. Run `npm run dev` to start the development server

## Technologies Used
- Frontend: React 18, Vite, Tailwind CSS, shadcn/ui
- Backend: Express.js, TypeScript
- Database: PostgreSQL with Drizzle ORM
- Styling: Tailwind CSS with custom color scheme
- Form Handling: React Hook Form with Zod validation

## Environment Variables Required
- DATABASE_URL - PostgreSQL connection string
- Other database connection variables (PGHOST, PGPORT, etc.)

Exported on: $(date)
EOF

# Create a simple installation script
cat > "$EXPORT_DIR/install.sh" << 'EOF'
#!/bin/bash
echo "Installing SiyanTransit Express..."
npm install
echo "Installation complete!"
echo "Configure your DATABASE_URL environment variable, then run:"
echo "npm run db:push"
echo "npm run dev"
EOF

chmod +x "$EXPORT_DIR/install.sh"

# Create archive
echo "Creating archive..."
tar -czf "$EXPORT_DIR.tar.gz" "$EXPORT_DIR"

echo "✓ Export completed!"
echo "✓ Directory: $EXPORT_DIR"
echo "✓ Archive: $EXPORT_DIR.tar.gz"
echo ""
echo "You can now:"
echo "1. Download the folder: $EXPORT_DIR"
echo "2. Download the archive: $EXPORT_DIR.tar.gz"
echo "3. Copy files manually from the directory structure"