import { useState, useEffect } from "react";
import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Search, Calculator, Truck } from "lucide-react";

export default function HeroSection() {
  const [trackingNumber, setTrackingNumber] = useState("");
  const [currentSlide, setCurrentSlide] = useState(0);

  const backgroundImages = [
    "/seaport operations.jpg", // Cargo containers at port
    "/Trading Services.jpg", // Trading services
    "/warehouse-interior-with-forklifts-shelving.jpg", // Warehouse operations
    "/IT Logistics Solutions.jpg"  // IT logistics solutions
  ];

  useEffect(() => {
    const timer = setInterval(() => {
      setCurrentSlide((prev) => (prev + 1) % backgroundImages.length);
    }, 4000);

    return () => clearInterval(timer);
  }, [backgroundImages.length]);

  const handleTrackShipment = () => {
    if (trackingNumber.trim()) {
      // TODO: Implement tracking functionality
      alert(`Tracking shipment: ${trackingNumber}`);
    }
  };

  return (
    <section className="relative bg-gradient-to-r from-primary to-secondary py-20 lg:py-32">
      {/* Sliding background images */}
      {backgroundImages.map((image, index) => (
        <div
          key={index}
          className={`absolute inset-0 bg-cover bg-center transition-opacity duration-1000 ${
            index === currentSlide ? "opacity-100" : "opacity-0"
          }`}
          style={{
            backgroundImage: `url('${image}')`
          }}
        />
      ))}
      
      {/* Enhanced background overlay with gradient */}
      <div className="absolute inset-0 bg-gradient-to-b from-black/40 via-black/30 to-black/50"></div>
      
      {/* Additional shadow overlay for better text contrast */}
      <div className="absolute inset-0 bg-black/20"></div>
      
      <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="text-center">
          <h1 className="text-4xl md:text-6xl font-bold text-white mb-6 drop-shadow-lg">
            Siyan Trans Express
            <span className="text-orange-400 drop-shadow-lg"> Logistics Solutions</span>
          </h1>
          <p className="text-xl md:text-2xl text-gray-100 mb-8 max-w-3xl mx-auto drop-shadow-md font-medium">
            One of the dynamic companies specialized in logistics services (transit, transport, handling) in the region, and Djibouti. Ensuring shipments are delivered quickly and safely to the door at destination using multi-modal transportation channels.
          </p>
          
          {/* Tracking Widget */}
          <div className="bg-white/90 backdrop-blur-sm rounded-xl shadow-xl p-6 md:p-8 max-w-2xl mx-auto mb-8 border border-white/20">
            <h3 className="text-2xl font-semibold text-gray-900 mb-4">Track Your Shipment</h3>
            <div className="flex flex-col md:flex-row gap-4">
              <Input
                type="text"
                placeholder="Enter tracking number"
                value={trackingNumber}
                onChange={(e) => setTrackingNumber(e.target.value)}
                className="flex-1 border-2 border-gray-200 focus:border-primary"
              />
              <Button
                onClick={handleTrackShipment}
                className="bg-primary hover:bg-secondary text-white px-8 py-3 shadow-lg hover:shadow-xl transition-all duration-200"
              >
                <Search className="h-4 w-4 mr-2" />
                Track Now
              </Button>
            </div>
          </div>

          {/* Primary CTAs */}
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button asChild size="lg" className="bg-orange-500 hover:bg-orange-600 text-white px-8 py-4 text-lg shadow-lg hover:shadow-xl transition-all duration-200 font-semibold">
              <Link href="/quotation">
                <Calculator className="h-5 w-5 mr-2" />
                Get Free Quote
              </Link>
            </Button>
            <Button asChild size="lg" variant="outline" className="bg-white/90 backdrop-blur-sm text-primary hover:bg-white hover:text-primary px-8 py-4 text-lg border-2 border-white shadow-lg hover:shadow-xl transition-all duration-200 font-semibold">
              <Link href="/services">
                <Truck className="h-5 w-5 mr-2" />
                Our Services
              </Link>
            </Button>
          </div>
        </div>
      </div>
    </section>
  );
}
